from . import file_helper, templates

import datetime
import os
import re
import sys
from xml.sax.saxutils import escape as xml_escape
from xml.sax.saxutils import quoteattr
import zipfile

prog = os.path.basename(sys.argv[0])

def which_tags_exist(html_text, tags):
    comment_l = '<!--'
    comment_r = '-->'
    search_items = [comment_l, comment_r] + [f'<{tag}' for tag in tags]
    regex = '|'.join([re.escape(i) for i in search_items])

    is_comment = False
    tags_found = set()
    for match in re.finditer(regex, html_text):
        match_str = match.group()
        if match_str == comment_l:
            is_comment = True
        elif match_str == comment_r:
            is_comment = False
        else:
            if not is_comment:
                tag_found = match_str[1:] # strip '<'
                tags_found.add(tag_found)

    return tags_found

def format_xml_element(tag, attr_dict, self_closed):
    result = f'<{tag}'
    for attr, value in attr_dict.items():
        if value is None:
            result += f' {attr}'
        else:
            result += f' {attr}={quoteattr(value)}'
    if self_closed:
        result += '/>'
    else:
        result += '>'

    return result

def format_item_and_write_file(z, opts, name, id, src=None):
    fpath = f'{opts.source_dir}/{name}'
    mime = file_helper.get_mimetype(name)
    item_attrs = {'href': name, 'id': id, 'media-type': mime}

    if mime != 'application/xhtml+xml':
        if src is None:
            z.write(fpath, name)
        else:
            f = z.open(name, 'w')
            f.write(src)
            f.close()

        return format_xml_element('item', item_attrs, True)

    # handle XHTML files

    # load XHTML source
    if src is None:
        with open(fpath, 'r') as f:
            src = f.read()

    # write to archive
    f = z.open(name, 'w')
    f.write(src.encode('utf-8'))
    f.close()

    # set properties attribute based on which tag exists
    tags = which_tags_exist(src, {'math', 'script', 'svg', 'nav'})
    tags_properties = {'math': 'mathml',
                       'script': 'scripted',
                       'svg': 'svg',
                       'nav': 'nav'}
    properties = {tags_properties[tag] for tag in tags}
    if len(properties) > 0:
        item_attrs['properties'] = ' '.join(properties)

    return format_xml_element('item', item_attrs, True)

def get_templated_xhtml(name, opts):
    fpath = f'{opts.source_dir}/{name}'
    with open(fpath, 'r') as f:
        file = f.read()

    stylelink_fmt = '<link rel=\'stylesheet\' href={name_a}/>'
    stylelinks = '\n'.join(stylelink_fmt.format(name_a=quoteattr(name))
                           for name in opts.stylesheets)

    html = opts.template_html.format(dc_title=xml_escape(opts.dc_title),
                                     stylelinks=stylelinks, file=file)
    xhtml = opts.template_xhtml.format(html=html,
        dc_title=xml_escape(opts.dc_title), stylelinks=stylelinks, file=file)
    return xhtml

def get_toc(opts):
    fpath = f'{opts.source_dir}/{opts.toc}'
    if os.path.exists(fpath):
        with open(fpath, 'r') as f:
            file = f.read()
    else:
        items = []
        for i, name in enumerate(opts.spine):
            if len(opts.toc_headings) == 0:
                heading = name
            elif i < len(opts.toc_headings):
                heading = opts.toc_headings[i]
            else:
                continue
            if heading  == '':
                continue

            name_a = quoteattr(name)
            heading_a = xml_escape(heading)
            item = f'<li><a href={name_a}>{heading_a}</a></li>'
            items.append(item)
        file = f'<ol>\n{'\n'.join(items)}\n</ol>'

    nav = opts.template_toc_nav.format(file=file)
    toc = opts.template_toc.format(nav=nav,
        dc_title=xml_escape(opts.dc_title), file=file)
    return toc

def format_manifest_items_and_write_files(z, opts):
    items = []

    # toc item
    toc_xhtml = get_toc(opts)
    item = format_item_and_write_file(z, opts, opts.toc, 'toc', src=toc_xhtml)
    items.append(item)

    for i, name in enumerate(opts.spine):
        if file_helper.get_extension(name).lower() == 'svg':
            item = format_item_and_write_file(z, opts, name, f'spine{i}')
        else:
            src = get_templated_xhtml(name, opts)
            item = format_item_and_write_file(z, opts, name, f'spine{i}',
                                              src=src)
        items.append(item)

    for i, name in enumerate(opts.resources):
        item = format_item_and_write_file(z, opts, name, f'res{i}')
        items.append(item)

    for i, name in enumerate(opts.stylesheets):
        item = format_item_and_write_file(z, opts, name, f'style{i}')
        items.append(item)

    return '\n'.join(items)

def timestamp_now():
    t = datetime.datetime.now(datetime.timezone.utc)
    return t.strftime('%Y-%m-%dT%H:%M:%SZ')

def spine_items(opts):
    items = []
    for i, name in enumerate(opts.spine):
        id_ = f'spine{i}'
        item = f'<itemref idref=\'{id_}\'></itemref>'
        items.append(item)

    return '\n'.join(items)

def write_package_opf(z, opts, manifest_items):
    timestamp = timestamp_now()

    dc_values = [('dc:coverage', opts.dc_coverage),
                 ('dc:creator', opts.dc_creator),
                 ('dc:date', timestamp if opts.dc_date_now
                             else opts.dc_date),
                 ('dc:description', opts.dc_description),
                 ('dc:publisher', opts.dc_publisher),
                 ('dc:rights', opts.dc_rights),
                 ('dc:source', opts.dc_source),
                 ('dc:subject', opts.dc_subject),
                 ('dc:type', opts.dc_type)]
    dc_values += [('dc:contributor', c) for c in opts.dc_contributor]
    dc_values += [('dc:relation', c) for c in opts.dc_relation]

    dc_extra = '\n'.join((f'<{tag}>{xml_escape(value)}</{tag}>'
                          for (tag, value) in dc_values
                          if value is not None))

    package = templates.package_opf.format(
        dc_identifier=xml_escape(opts.dc_identifier),
        dc_title=xml_escape(opts.dc_title),
        dc_language=xml_escape(opts.dc_language),
        dc_format=xml_escape(templates.mimetype),
        dc_extra=dc_extra,
        dcterms_modified=xml_escape(timestamp),
        manifestitems=manifest_items,
        spineitems=spine_items(opts),
    )

    f = z.open('package.opf', 'w')
    f.write(package.encode('utf-8'))
    f.close()

def generate_epub(opts):
    z = zipfile.ZipFile(opts.output, 'w', zipfile.ZIP_STORED)
    z.writestr('mimetype', templates.mimetype)
    z.close()

    z = zipfile.ZipFile(opts.output, 'a', zipfile.ZIP_DEFLATED)
    z.writestr('META-INF/container.xml', templates.container_xml)

    manifest_items = format_manifest_items_and_write_files(z, opts)
    write_package_opf(z, opts, manifest_items)
    z.close()
