from . import epub, file_helper, templates

import argparse
import collections
import datetime
import os
import sys
import uuid

prog = os.path.basename(sys.argv[0])

def get_and_validate_options():
    parser = argparse.ArgumentParser(prog=prog, usage=(f'{prog} [-h|--help] '
        +'--output OUTPUT --spine NAME... [OPTION...]'))

    # file structure
    parser.add_argument('--output', '-o', type=str, required=True,
        help='Path to the generated ePub file.')
    parser.add_argument('--spine', '-s', type=str, nargs='+', required=True,
        help='Names of files for sections/chapters. Must be XHTML or SVG.')
    parser.add_argument('--resources', type=str, nargs='+', default=[],
        help='Names of files to be embedded as media/resources.')
    parser.add_argument('--stylesheets', type=str, nargs='+', default=[],
        help='Names of stylesheet files. Must be CSS.')
    parser.add_argument('--toc', type=str, default='toc.xhtml',
        help='Name of table of contents file. Must be XHTML.')
    parser.add_argument('--source-dir', '--src-dir', '-d',
                        type=str, default='.',
                        help='Directory to look for source files in.')

    # document metadata
    parser.add_argument('--dc-title', type=str, default='Untitled',
        help='Title of generated ePub document.')
    parser.add_argument('--dc-identifier', type=str, default=None,
        help='Document identifier, either a URL or RFC 8141 NID.')
    parser.add_argument('--dc-language', type=str, default='en',
        help='Document language, as an RFC 5646 identifier.')
    parser.add_argument('--dc-creator', type=str,
        help='Document author.')
    parser.add_argument('--dc-contributor', type=str, nargs='+', default=[],
        help='List of document non-author contributors.')
    parser.add_argument('--dc-coverage', type=str,
        help='Document coverage.')
    parser.add_argument('--dc-date', type=str,
        help=('Document date, in ISO 8601 format. Assumed to be UTC '
              +'if timezone not specified.'))
    parser.add_argument('--dc-date-now', action='store_true',
        help='Set document date to the current time.')
    parser.add_argument('--dc-description', type=str,
        help='Document description.')
    parser.add_argument('--dc-publisher', type=str,
        help='Document publisher.')
    parser.add_argument('--dc-relation', type=str, nargs='+', default=[],
        help='List of URIs (or other formal identifiers) of related resources.')
    parser.add_argument('--dc-rights', type=str,
        help='Document copyright.')
    parser.add_argument('--dc-source', type=str,
        help='Document source.')
    parser.add_argument('--dc-subject', type=str,
        help='Document subject.')
    parser.add_argument('--dc-type', type=str,
        help='Document type.')
    parser.add_argument('--toc-headings', type=str, nargs='+', default=[],
        help='Headings in table of contents for files in --spine')

    # templating
    parser.add_argument('--template-xhtml', type=str, default=templates.xhtml,
        help='Template for root of XHTML files.')
    parser.add_argument('--template-html', type=str, default=templates.html,
        help='Template for the contents of the <html> tag in XHTML files.')
    parser.add_argument('--template-toc', type=str, default=templates.toc,
        help='Template for the root of the table of contents XHTML file.')
    parser.add_argument('--template-toc-nav', type=str, default=templates.nav,
        help=('Template for the contents of the <nav> tag in the '
              +'table of contents XHTML file.'))

    opts = parser.parse_args()

    # make sure files are not duplicated

    all_files = sorted(opts.spine + opts.resources
                       + opts.stylesheets + [opts.toc])
    for a, b in zip(all_files[:-1], all_files[1:]):
        if a == b:
            if a == opts.toc:
                print(f'{prog}: error: table of contents filename \'{a}\' ',
                      end='', file=sys.stderr)
                print('is duplicated. Use --toc to change it.',
                      file=sys.stderr)
            else:
                print(f'{prog}: error: filename \'{a}\' is duplicated.',
                file=sys.stderr)
            exit(-1)

    # make sure files do not have reserved names

    if any((name in {'mimetype', 'package.opf'} or name.startswith('META-INF'))
           for name in all_files):
        print((f'{prog}: error: file names `mimetype`, `META-INF`, and '
               +'`package.opf` are reserved.'), file=sys.stderr)
        exit(-1)

    # make sure files have valid types

    if file_helper.get_extension(opts.toc).lower() != 'xhtml':
        print(f'{prog}: error: --toc must be XHTML type.',
              file=sys.stderr)
        exit(-1)

    for name in opts.spine:
        if file_helper.get_extension(name).lower() not in {'xhtml', 'svg'}:
            print(f'{prog}: error: file \'{name}\' had invalid type ', end='',
                  file=sys.stderr)
            print('for --spine. Valid types are XHTML and SVG.',
                  file=sys.stderr)
            exit(-1)

    valid_extensions = file_helper.mimetypes.keys()
    for name in opts.resources:
        if file_helper.get_extension(name).lower() not in valid_extensions:
            print(f'{prog}: error: file \'{name}\' had invalid type ', end='',
                  file=sys.stderr)
            print(f'for --resources. Valid types are: ',
                  end='', file=sys.stderr)
            print(' '.join(valid_extensions).upper(), file=sys.stderr)
            exit(-1)

    for name in opts.stylesheets:
        if file_helper.get_extension(name).lower() != 'css':
            print(f'{prog}: error: file \'{name}\' had invalid type ', end='',
                  file=sys.stderr)
            print('for --stylesheets; must be CSS.', file=sys.stderr)
            exit(-1)

    # make sure files exist

    for name in opts.spine + opts.resources + opts.stylesheets:
        fpath = f'{opts.source_dir}/{name}'
        if not os.path.exists(fpath):
            print(f'{prog}: error: file \'{fpath}\' does not exist',
                  file=sys.stderr)
            exit(-1)

    # validate date

    if opts.dc_date is not None:
        if opts.dc_date_now:
            print(f'{prog}: error: --dc-date and --dc-date-now ', end='',
                  file=sys.stderr)
            print('must not be set simultaneously.', file=sys.stderr)
            exit(-1)

        # try to read date
        try:
            t = datetime.datetime.fromisoformat(opts.dc_date)
        except ValueError:
            print(f'{prog}: error: --dc-date must be in ISO 8601 format.',
                  file=sys.stderr)
            exit(-1)

        if t.tzinfo is None:
            t = t.replace(tzinfo=datetime.timezone.utc)
        t_utc = t.astimezone(datetime.timezone.utc)
        opts.dc_date = t_utc.strftime('%Y-%m-%dT%H:%M:%SZ')

    # validate toc options

    if len(opts.toc_headings) > 0:
        toc_fpath = f'{opts.source_dir}/{opts.toc}'
        if os.path.exists(toc_fpath):
            print((f'{prog}: error: --toc-headings must not be set '
                   +'if --toc exists in --source-dir.'), file=sys.stderr)
            exit(-1)

        if max(len(heading) for heading in opts.toc_headings) == 0:
            print((f'{prog}: error: at least one of --toc-headings '+
                   'must be non-empty'), file=sys.stderr)
            exit(-1)

    if len(opts.toc_headings) > len(opts.spine):
        print(f'{prog}: error: --toc-headings longer than --spine.',
              file=sys.stderr)
        exit(-1)

    # if no identifier specified, pick a random one

    if opts.dc_identifier is None:
        opts.dc_identifier = f'urn:uuid:{uuid.uuid4()}'
        print(f'{prog}: warning: --dc-identifier not set. ',
              end='', file=sys.stderr)
        print(f'Using random identifier \'{opts.dc_identifier}\', ',
              end='', file=sys.stderr)
        print('but it is strongly recommended to set a persistent identifier.',
              file=sys.stderr)

    return opts

def main():
    opts = get_and_validate_options()
    epub.generate_epub(opts)

if __name__ == '__main__':
    main()
