import build_system
import blog

import datetime
import json

rfc_2822_fmt = '%a, %d %b %Y %H:%M:%S %z'
max_items_bytes = 100_000
max_items = 25

def _items_from_blog_posts(blog_data_fpath):
    blog_posts = blog.get_blog_post_info(blog_data_fpath)
    items = []
    dates = []
    for post in blog_posts:
        title = post['title']
        itemtitle = f'Blog post: {title}'
        url = post['url']
        fullurl = f'https://marianicolae.com{url}'
        date = post['date']
        item = {'title': itemtitle,
                'url': fullurl,
                'description': post['description'],
                'date': date.strftime(rfc_2822_fmt)}
        items.append(item)
        dates.append(date)

    return items, dates

def _load_items(items_fpath):
    with open(items_fpath, 'r') as f:
        raw_items = json.load(f)

    items = []
    dates = []
    for item in raw_items:
        date = datetime.datetime.fromisoformat(item['date'])
        date_str = date.strftime(rfc_2822_fmt)
        item['date'] = date_str
        items.append(item)
        dates.append(date)

    return items, dates

def _rss_func(build_fpath, feed_xmlfmt_fpath,
              item_xmlfmt_fpath, items_fpath, blog_data_fpath):
    with open(feed_xmlfmt_fpath, 'r') as f:
        feed_xmlfmt = f.read()
    with open(item_xmlfmt_fpath, 'r') as f:
        item_xmlfmt = f.read()

    blog_post_items, blog_post_dates = _items_from_blog_posts(blog_data_fpath)
    other_items, other_dates = _load_items(items_fpath)
    items = blog_post_items + other_items
    dates = blog_post_dates + other_dates

    sorted_dates, sorted_items = zip(*sorted(zip(dates, items), reverse=True))

    items_xml = ''
    for item in sorted_items[:max_items]:
        item_xml = item_xmlfmt.format(**item)
        new_items_xml = items_xml + item_xml
        if len(new_items_xml) > max_items_bytes:
            break
        items_xml = new_items_xml

    ... # get years, now_str
    min_year = dates[-1].year
    max_year = dates[0].year
    if min_year == max_year:
        years = f'{max_year}'
    else:
        years = f'{min_year}-{max_year}'
    now = datetime.datetime.now(datetime.timezone.utc)
    now_str = now.strftime(rfc_2822_fmt)
    feed_xml = feed_xmlfmt.format(years=years, date=now_str, items=items_xml)

    with open(build_fpath, 'w') as f:
        f.write(feed_xml)

def add_rss_rule(b, target, feed_xmlfmt_fpath,
                 item_xmlfmt_fpath, items_fpath, blog_data_fpath):
    r = b.add_rule(target,
            feed_xmlfmt_fpath, item_xmlfmt_fpath, items_fpath, blog_data_fpath)
    r.add_function(_rss_func, target, feed_xmlfmt_fpath,
                   item_xmlfmt_fpath, items_fpath, blog_data_fpath)
    b.add_deps('build', target)
