import os
import subprocess
import time

class Rule:
    def __init__(self, target, *deps, phony=False):
        self.target = target
        self.deps = deps
        self.is_phony = phony
        self._steps = list()

    def add_command(self, cmd):
        self._steps.append(('cmd', cmd))

    def add_command_args(self, *args):
        self._steps.append(('args', args))

    def add_function(self, func, *args, **kwargs):
        self._steps.append(('func', (func, args, kwargs)))

    def run_recipe(self):
        if len(self._steps) == 0:
            return

        print(f'Making `{self.target}`')
        for typ, step in self._steps:
            if typ == 'cmd':
                cmd = step
                status = os.system(cmd)
            elif typ == 'args':
                args = step
                result = subprocess.run(args)
                status = result.returncode
            elif typ == 'func':
                func, args, kwargs = step
                func(*args, **kwargs)
                status = 0

            if status != 0:
                raise RuntimeError(f'Error when making `{self.target}`')

class Build:
    def __init__(self):
        self._rules = dict()
        self._leaves = set()

    def add_rule(self, target, *deps, phony=False):
        if target in self._rules:
            raise ValueError(f'Target `{target}` already exists.')

        rule = Rule(target, *deps, phony=phony)
        self._rules[target] = rule

        if target in self._leaves:
            self._leaves.remove(target)
        elif len(deps) == 0:
            self._leaves.add(target)

        for dep in deps:
            if dep not in self._rules:
                self._leaves.add(dep)

        return rule

    def add_deps(self, target, *deps):
        if target not in self._rules:
            raise ValueError(f'Target {target} does not exist.')
        self._rules[target].deps += deps

    def _get_subgraph(self, root_target):
        if root_target not in self._rules:
            raise ValueError(f'Target `{root_target}` does not exist.')

        nodes = set()
        rules = dict()
        leaves = set()
        def dfs_recursive(node):
            nodes.add(node)

            if node not in self._rules:
                leaves.add(node)
                return

            target = node
            rule = self._rules[node]
            rules[target] = rule

            deps = rule.deps
            if len(deps) == 0:
                leaves.add(node)
                return

            for dep in deps:
                if dep not in nodes:
                    dfs_recursive(dep)

        dfs_recursive(root_target)
        return rules, leaves

    def _topological_sort(self, root_target):
        # implements Kahn's algorithm

        rules, leaves = self._get_subgraph(root_target)
        
        dependencies = dict()
        dependents = dict()
        for target, rule in rules.items():
            dependencies[target] = set(rule.deps)
            for dep in rule.deps:
                if dep not in dependents:
                    dependents[dep] = set()
                dependents[dep].add(target)

        sort = list()
        while len(leaves) > 0:
            node = leaves.pop()
            sort.append(node)
            
            if node not in dependents:
                continue

            for dependent in dependents[node]:
                dependencies[dependent].remove(node)
                if len(dependencies[dependent]) == 0:
                    dependencies.pop(dependent)
                    leaves.add(dependent)

            dependents.pop(node)

        if len(dependents) > 0:
            raise ValueError('Dependency graph is not acyclic.')

        return sort

    def make(self, target):
        if target not in self._rules:
            raise ValueError(f'Target `{target}` not found.')

        topological_sort = self._topological_sort(target)

        subgraph_date_modified = dict()
        for node in topological_sort:
            if node not in self._rules:
                subgraph_date_modified[node] = os.path.getmtime(node)
                continue

            target = node
            rule = self._rules[target]
            deps = rule.deps

            if rule.is_phony or not os.path.exists(target):
                rule.run_recipe()
                subgraph_date_modified[target] = time.time()
                continue

            date_modified = os.path.getmtime(target)
            max_dep_date_modified = max(subgraph_date_modified[dep]
                                        for dep in deps)
            if date_modified <= max_dep_date_modified:
                rule.run_recipe()
                subgraph_date_modified[target] = time.time()
            else:
                subgraph_date_modified[target] = date_modified
