import site_build

import pathlib

_navigation_pages = []

def add_navigation(*page_titles):
    global _navigation_pages
    _navigation_pages += list(page_titles)

def _generate_navigation(title):
    navigation_htmls = []
    for page_title in _navigation_pages:
        if title == page_title:
            navigation_htmls.append(title)
        else:
            urlpath = site_build.pages[page_title]
            link = f'<a href=\'{urlpath}\'>{page_title}</a>'
            navigation_htmls.append(link)
    return ' | '.join(navigation_htmls)

_boilerplate_fpath = 'src/boilerplate.htmlfmt'
_boilerplate = None
def _format_boilerplate(title, body, years):
    global _boilerplate
    if _boilerplate is None:
        with open(_boilerplate_fpath, 'r') as f:
            _boilerplate = f.read()

    navigation = _generate_navigation(title)
    output = _boilerplate.format(title=title, navigation=navigation,
                                 body=body, years=years)
    return output

def _path_open_write(fpath):
    path = pathlib.Path(fpath)
    path.parent.mkdir(parents=True, exist_ok=True)
    return open(fpath, 'w')

# rule for simple webpage consisting of body html read from a file
def simple_rule(build, page_title, page_path, page_src, years=''):
    fpath = site_build.page_fpath(page_path)

    def rule_func():
        with open(page_src, 'r') as f:
            body_html = f.read()
        full_html = _format_boilerplate(page_title, body_html, years)
        with _path_open_write(fpath) as f:
            f.write(full_html)

    r = site_build.page_rule(build, page_title, page_path,
                             _boilerplate_fpath, page_src)
    r.add_function(rule_func)

def _clear_file(fpath):
    f = _path_open_write(fpath)
    f.close()

# rule for webpage whose body HTML is generated by Python functions
def generated_rule(build, page_title, page_path, *deps, years=''):
    class RuleWrapper:
        def __init__(self, rule, page_title):
            self._rule = rule
            self._page_title = page_title
            self.target = rule.target
            self.deps = deps
            self.is_phony = False

            self._rule.add_function(_clear_file, self.target)

        def add_function(self, func, *args, **kwargs):
            def rule_func():
                body_html = func(*args, **kwargs)
                full_html = _format_boilerplate(self._page_title, body_html,
                                                years)
                with open(self.target, 'w+') as f:
                    f.write(full_html)

            self._rule.add_function(rule_func)

    rule = site_build.page_rule(build, page_title, page_path,
                                _boilerplate_fpath, *deps)
    return RuleWrapper(rule, page_title)
