import build_system
import boilerplate
import helper
import site_build

import datetime
import html
import json
import os
import subprocess

listpage_htmlfmt_fpath = 'src/software/listpage.htmlfmt'
listing_htmlfmt_fpath = 'src/software/listing.htmlfmt'
project_page_htmlfmt_fpath = 'src/software/project_page.htmlfmt'
changelog_section_htmlfmt_fpath = 'src/software/changelog_section.htmlfmt'
readme_section_htmlfmt_fpath = 'src/software/readme_section.htmlfmt'
versions_page_htmlfmt_fpath = 'src/software/versions_page.htmlfmt'
projects_json_fpath = 'data/software/auto/projects.json'
versions_json_fpath_fmt = 'data/software/auto/projects/{name}/versions.json'
readme_fpath_fmt = 'data/software/auto/projects/{name}/README.md'
changelog_fpath_fmt = 'data/software/auto/projects/{name}/CHANGELOG.md'
projects_info_json_fpath = 'data/software/project_info.json'
style_fpath = 'src/software/style.css'

date_raw_fmt = '%Y-%m-%dT%H:%M:%S%z'
date_pretty_fmt = '%Y-%m-%d'

_project_info = None
def get_projects_info():
    global _project_info
    if _project_info is not None:
        return _project_info

    with open(projects_json_fpath, 'r') as f:
        project_info_raw = json.load(f)

    with open(projects_info_json_fpath, 'r') as f:
        project_info_extra = json.load(f)

    _project_info = dict()
    for name in project_info_raw.keys():
        _project_info[name] = dict()
        for k in ['date_published', 'date_modified']:
            _project_info[name][k] = datetime.datetime.fromisoformat(
                project_info_raw[name][k])

        extra_data_default = {'description': None, 'title': name}
        if name in project_info_extra:
            for k, default_v in extra_data_default.items():
                if k in project_info_extra[name]:
                    _project_info[name][k] = project_info_extra[name][k]
                else:
                    _project_info[name][k] = default_v
        else:
            _project_info[name].update(extra_data_default)

    for name, info in _project_info.items():
        info['url'] = f'/software/{name}/'
        info['name'] = name

    return _project_info

def _project_list_func(listpage_htmlfmt_fpath,
                       listing_htmlfmt_fpath, project_info):
    with open(listpage_htmlfmt_fpath, 'r') as f:
        listpage_htmlfmt = f.read()
    with open(listing_htmlfmt_fpath, 'r') as f:
        listing_htmlfmt = f.read()

    sorted_project_info = sorted(
        ((name, info) for (name, info) in project_info.items()),
        key = lambda nameinfo: nameinfo[1]['date_modified'], reverse=True)

    listings = []
    for name, info in sorted_project_info:
        title = html.escape(info['title'])
        description = (f'{html.escape(info['description'])}<br>'
                       if info['description'] is not None else '')
        date_raw = info['date_modified'].strftime(date_raw_fmt)
        date_pretty = info['date_modified'].strftime(date_pretty_fmt)
        listing = listing_htmlfmt.format(
            name=name, title=title, description=description,
            date_raw=date_raw, date_pretty=date_pretty)
        listings.append(listing)

    listings_html = '\n'.join(listings)

    listpage_html = listpage_htmlfmt.format(listing=listings_html)
    return listpage_html

def _markdown_to_html(md_text):
    result = subprocess.run(['pandoc', '--from', 'markdown', '--to', 'html'],
        input=md_text.encode('utf-8'), capture_output=True)
    output_bytes = result.stdout
    output = output_bytes.decode('utf-8')
    return output

def _release_table(project_info, versions_info):
    name = project_info['name']

    table_headings = ['Version', 'Date', 'Download Links']
    table_header = ''.join(f'<th>{cell}</th>' for cell in table_headings)

    versions = []
    for tag, version_info in versions_info.items():
        version_dict = {
            'date': datetime.datetime.fromisoformat(version_info['date']),
            'tag': tag}
        versions.append(version_dict)

    sorted_versions = sorted(versions, key=lambda v: v['date'], reverse=True)

    table_rows = [table_header]
    for i, version_info in enumerate(sorted_versions):
        tag = version_info['tag']
        version_str = f'{tag} (latest)' if i == 0 else tag
        date_str = version_info['date'].strftime(date_pretty_fmt)
        links = []
        for extension in ['tar.gz', 'zip']:
            basename = f'{name}-{tag}.{extension}'
            url = f'/files/software/{basename}'
            link = f'<a href=\'{url}\' download><code>{basename}</code></a>'
            links.append(link)
        links_str = '<br>'.join(links)
        table_cells = [version_str, date_str, links_str]

        table_row = ''.join(f'<td>{cell}</td>' for cell in table_cells)
        table_rows.append(table_row)

    table_core_html = '\n'.join(f'<tr>{row}</tr>'for row in table_rows)
    table_html = f'<table class=\'versions\'>\n{table_core_html}\n</table>'
    return table_html

def _project_page_func(project_page_htmlfmt_fpath, versions_json_fpath,
    readme_fpath, readme_section_htmlfmt_fpath, changelog_fpath,
    changelog_section_htmlfmt_fpath, project_info):

    with open(project_page_htmlfmt_fpath, 'r') as f:
        project_page_htmlfmt = f.read()
    with open(versions_json_fpath, 'r') as f:
        versions_info = json.load(f)

    name = project_info['name']
    title = html.escape(project_info['title'])
    latest = max(versions_info.keys(), key=lambda v: versions_info[v]['date'])

    release_table = _release_table(project_info, versions_info)

    with open(readme_fpath, 'r') as f:
        readme_md = f.read()
    readme_html = _markdown_to_html(readme_md)
    readme_html_escaped = readme_html.replace('\\', '\\\\')
    with open(readme_section_htmlfmt_fpath, 'r') as f:
        readme_section_htmlfmt = f.read()
    readme_section = readme_section_htmlfmt.format(
        html=readme_html_escaped)

    with open(changelog_fpath, 'r') as f:
        changelog_md = f.read()
    changelog_html = _markdown_to_html(changelog_md)
    changelog_html_escaped = changelog_html.replace('\\', '\\\\')
    with open(changelog_section_htmlfmt_fpath, 'r') as f:
        changelog_section_htmlfmt = f.read()
    changelog_section = changelog_section_htmlfmt.format(
        html=changelog_html_escaped)

    project_page_html = project_page_htmlfmt.format(name=name, title=title,
        readme=readme_section, changelog=changelog_section, latest=latest,
        release_table=release_table)
    return project_page_html

def add_software_rules(build, pagetitle):
    style_url = '/software-style.css'
    r = build.add_rule(f'build{style_url}', style_fpath)
    r.add_command_args('cp', r.deps[0], r.target)
    build.add_deps('build', r.target)
    head = f'<link rel=\'stylesheet\' type=\'text/css\' href=\'{style_url}\'>'

    project_info = get_projects_info()
    project_dates = [info['date_published'] for info in project_info.values()]
    project_dates += [info['date_modified'] for info in project_info.values()]
    years = helper.year_range(*project_dates)
    
    r = boilerplate.generated_rule(build, pagetitle, '/software/',
        listpage_htmlfmt_fpath, listing_htmlfmt_fpath, projects_json_fpath,
        projects_info_json_fpath, years=years)
    r.add_function(_project_list_func, listpage_htmlfmt_fpath,
        listing_htmlfmt_fpath, project_info)

    for name, info in project_info.items():
        pagetitle = f'{info['title']} - Software'
        url = info['url']
        years = helper.year_range(info['date_published'], info['date_modified'])
        versions_json_fpath = versions_json_fpath_fmt.format(name=name)
        readme_fpath = readme_fpath_fmt.format(name=name)
        changelog_fpath = changelog_fpath_fmt.format(name=name)
        r = boilerplate.generated_rule(build, pagetitle, url,
            project_page_htmlfmt_fpath, projects_json_fpath,
            projects_info_json_fpath, versions_json_fpath, readme_fpath,
            readme_section_htmlfmt_fpath, changelog_fpath,
            changelog_section_htmlfmt_fpath, years=years, head_text=head)
        r.add_function(_project_page_func, project_page_htmlfmt_fpath,
            versions_json_fpath, readme_fpath, readme_section_htmlfmt_fpath,
            changelog_fpath, changelog_section_htmlfmt_fpath, info)
