import build_system
import blog
import helper
import software

import datetime
import json
from xml.sax.saxutils import escape as xml_escape

feed_xmlfmt_fpath = 'src/rss/feed.xmlfmt'
item_xmlfmt_fpath = 'src/rss/item.xmlfmt'
items_fpath = 'src/rss/items.json'
software_target_fmt = 'build/software/{name}/rss.xml'

rfc_2822_fmt = '%a, %d %b %Y %H:%M:%S %z'
max_items_bytes = 100_000
max_items = 25

def _items_from_blog_posts(blog_data_fpath):
    blog_posts = blog.get_blog_post_info(blog_data_fpath)
    items = []
    dates = []
    for post in blog_posts:
        title = post['title']
        itemtitle = f'Blog post: {title}'
        url = post['url']
        fullurl = f'https://marianicolae.com{url}'
        date = post['date']
        item = {'title': itemtitle,
                'url': fullurl,
                'description': post['description'],
                'date': date.strftime(rfc_2822_fmt)}
        items.append(item)
        dates.append(date)

    return items, dates

def _items_from_software(project_info):

    items = []
    dates = []
    for name, info in project_info.items():
        if 'title' in info.keys():
            title = info['title']
        else:
            title = name
        itemtitle = f'Software: {title}'
        url = f'https://marianicolae.com/software/{name}/'
        date = info['date_published']
        description = f'A new software project \'{title}\' '
        description += 'has been published.'
        if 'description' in info.keys():
            description += f'<br/>Description: {info['description']}'
        item = {'title': itemtitle,
                'url': url,
                'description': description,
                'date': date.strftime(rfc_2822_fmt)}
        items.append(item)
        dates.append(date)

    return items, dates

def _load_items(items_fpath):
    with open(items_fpath, 'r') as f:
        raw_items = json.load(f)

    items = []
    dates = []
    for item in raw_items:
        date = datetime.datetime.fromisoformat(item['date'])
        date_str = date.strftime(rfc_2822_fmt)
        item['date'] = date_str
        items.append(item)
        dates.append(date)

    return items, dates

_feed_xmlfmt = None
_item_xmlfmt = None

def _feed_xml(items, dates, feed_xmlfmt_fpath, item_xmlfmt_fpath,
              title, url, description, source, source_url):
    global _feed_xmlfmt
    if _feed_xmlfmt is None:
        with open(feed_xmlfmt_fpath, 'r') as f:
            _feed_xmlfmt = f.read()

    global _item_xmlfmt
    if _item_xmlfmt is None:
        with open(item_xmlfmt_fpath, 'r') as f:
            _item_xmlfmt = f.read()

    sorted_dates, sorted_items = zip(*sorted(zip(dates, items), reverse=True))

    items_xml = ''
    for item in sorted_items[:max_items]:
        item_xml = _item_xmlfmt.format(
            **{k: xml_escape(v) for (k, v) in item.items()},
            source=xml_escape(source),
            source_url=source_url)
        new_items_xml = items_xml + item_xml
        if len(new_items_xml) > max_items_bytes:
            break
        items_xml = new_items_xml

    years = helper.year_range(*dates)
    now = datetime.datetime.now(datetime.timezone.utc)
    now_str = now.strftime(rfc_2822_fmt)
    feed_xml = _feed_xmlfmt.format(title=title, url=url,
        description=description, years=years, date=now_str, items=items_xml)
    return feed_xml

def _rss_func(build_fpath, feed_xmlfmt_fpath, item_xmlfmt_fpath,
    items_fpath, blog_data_fpath, project_info):

    blog_post_items, blog_post_dates = _items_from_blog_posts(blog_data_fpath)
    software_items, software_dates = _items_from_software(project_info)
    other_items, other_dates = _load_items(items_fpath)
    items = blog_post_items + software_items + other_items
    dates = blog_post_dates + software_dates + other_dates

    feed_xml = _feed_xml(items, dates, feed_xmlfmt_fpath, item_xmlfmt_fpath,
        title='Maria Nicolae\'s Website',
        url='https://marianicolae.com/',
        description='RSS feed for Maria Nicolae\'s personal website.',
        source='Maria Nicolae\'s Website',
        source_url='https://marianicolae.com/rss.xml')

    f = helper.path_open_write(build_fpath)
    f.write(feed_xml)
    f.close()

def _software_feed_func(build_fpath, name, title, feed_xmlfmt_fpath,
                        item_xmlfmt_fpath, versions_json_fpath):
    with open(versions_json_fpath, 'r') as f:
        versions_info = json.load(f)

    items = []
    dates = []
    for version, info in versions_info.items():
        date = datetime.datetime.fromisoformat(info['date'])
        itemtitle = f'{title} updated to {version}'
        url = f'https://marianicolae.com/software/{name}/'
        description = f'{title} has been updated to {version}. '
        description += 'Release archives are available for download:'
        for extension in ['tar.gz', 'zip']:
            archive_name = f'{name}-{version}.{extension}'
            archive_url_path = 'https://marianicolae.com/files/software'
            archive_url = f'{archive_url_path}/{archive_name}'
            description += f'<br/><a href=\'{archive_url}\'>{archive_name}</a>'
        item = {'title': itemtitle,
                'url': url,
                'description': description,
                'date': date.strftime(rfc_2822_fmt)}
        items.append(item)
        dates.append(date)

    feed_xml = _feed_xml(items, dates, feed_xmlfmt_fpath, item_xmlfmt_fpath,
        title=f'{title} Updates',
        url=f'https://marianicolae.com/software/{name}/',
        description=f'RSS feed for updates to {title}.',
        source=f'{title} Updates',
        source_url=f'https://marianicolae.com/software/{name}/')

    f = helper.path_open_write(build_fpath)
    f.write(feed_xml)
    f.close()

def add_rss_rule(build):
    projects_info = software.get_projects_info()

    r = build.add_rule('build/rss.xml',
                       feed_xmlfmt_fpath, item_xmlfmt_fpath, items_fpath,
                       blog.blog_data_fpath, software.projects_json_fpath,
                       software.projects_info_json_fpath)
    r.add_function(_rss_func, r.target, feed_xmlfmt_fpath, item_xmlfmt_fpath,
                   items_fpath, blog.blog_data_fpath, projects_info)
    build.add_deps('build', r.target)

    for name, info in projects_info.items():
        build_fpath = software_target_fmt.format(name=name)
        if 'title' in info.keys():
            title = info['title']
        else:
            title = name
        versions_json_fpath = software.versions_json_fpath_fmt.format(
            name=name)

        r = build.add_rule(build_fpath, feed_xmlfmt_fpath, item_xmlfmt_fpath,
                       software.projects_info_json_fpath, versions_json_fpath)
        r.add_function(_software_feed_func, build_fpath, name, title,
            feed_xmlfmt_fpath, item_xmlfmt_fpath, versions_json_fpath)
        build.add_deps('build', build_fpath)
